/*****************************************************************************

       Copyright 1993, 1994, 1995  Digital Equipment Corporation,
		       Maynard, Massachusetts.

			All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, provided  
that the copyright notice and this permission notice appear in all copies  
of software and supporting documentation, and that the name of Digital not  
be used in advertising or publicity pertaining to distribution of the software 
without specific, written prior permission. Digital grants this permission 
provided that you prominently mark, as not part of the original, any 
modifications made to this software or documentation.

Digital Equipment Corporation disclaims all warranties and/or guarantees  
with regard to this software, including all implied warranties of fitness for 
a particular purpose and merchantability, and makes no representations 
regarding the use of, or the results of the use of, the software and 
documentation in terms of correctness, accuracy, reliability, currentness or
otherwise; and you rely on the software, documentation and results solely at 
your own risk. 

******************************************************************************/
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Tsunami management library
 * Adapted from EBSDK 28 June 1999 by Stig Telfer, Alpha Processor, Inc.
 */

#include "lib.h"
#include "pci.h"
#include "memory.h"

#include "uilib.h"
#include "northbridge.h"
#include "northbridge/tsunami.h"


/*----------------------------------------------------------------------*/
/* Machine configuration data */

/* number of independent PCI bus trees */
const unsigned char nb_pci_hosecount=2;


/*----------------------------------------------------------------------*/
/* IO space and CSR managment */


tsunami_csr *pchip0 = (tsunami_csr *)PCHIP0_CSR;
tsunami_csr *pchip1 = (tsunami_csr *)PCHIP1_CSR;
tsunami_csr *cchip = (tsunami_csr *)CCHIP_CSR;


static void WriteIOB(ul address,ub value) {WriteB(address, value); }
static void WriteIOW(ul address,uw value) {WriteW(address, value); }
static void WriteIOL(ul address,ui value) {WriteL(address, value); }
static void WriteIOQ(ul address,ul value) {WriteQ(address, value); }

static ub ReadIOB(ul address) { return(ReadB(address)); }
static uw ReadIOW(ul address) { return(ReadW(address)); }
static ui ReadIOL(ul address) { return(ReadL(address)); }
static ul ReadIOQ(ul address) { return(ReadQ(address)); }



/*----------------------------------------------------------------------*/
/* Control/Analyse the Tsunami CSRs for SERR/ECC error conditions */

void nb_ecc( unsigned o )		/* o = 'on' or 'off' */
{
}


/* NB this function is DISUSED: soon to disappear? */

int nb_eccstat( char *buf )
{
    return 0;
}



/*----------------------------------------------------------------------*/
/* nb_setup - early initialisations relating to the northbridge. */ 

void nb_setup( void )
{

    const unsigned long errmask =	perror_m_serr	| 
					perror_m_perr	|
					perror_m_dcrto	|
					perror_m_sge	|
					perror_m_ape	|
#if 0			/* don't want target aborts */
					perror_m_ta	|
#endif
					perror_m_rdpe	|
					perror_m_uecc	|
					perror_m_cre;

    /* First show we're about to probe PCI */
    diags_subsys_stat( SYS_PCI, DEV_PROBING );

    PCIInit();


    /* clear any current error state in the Pchips by writing 1's to the error
     * bits, except for bit 9 which is MBZ */
    pchip0[ PERROR ].r = 0xDFFUL;
    pchip1[ PERROR ].r = 0xDFFUL;
    mb();					/* make sure they write */

    /* enable all the Pchip error traps that we want to respond to */

    pchip0[ PERRMASK ].r = errmask;		/* our error types enabled */
    pchip1[ PERRMASK ].r = errmask;		/* our error types enabled */
    mb();

    diags_subsys_stat( SYS_PCI, DEV_SETUP );
}



/*----------------------------------------------------------------------*/
/* nb_fixup - late initialisations and fixups */

void nb_fixup( void )
{

    /* Establish DMA windows. */

#if (PCI_BASE_1_USED)
    pchip0[WSM0].r = PCI_MASK_1;
    pchip0[TBA0].r = PCI_TBASE_1;
    pchip0[WSBA0].r= PCI_BASE_1 | 1;

    /* WriteIOQ((((ul) PCHIP0_CSR<<24) | PCHIP0_WSM0), PCI_MASK_1); */
    /* WriteIOQ((((ul) PCHIP0_CSR<<24) | PCHIP0_TBA0), PCI_TBASE_1); */
    /* WriteIOQ((((ul) PCHIP0_CSR<<24) | PCHIP0_WSBA0), PCI_BASE_1 | 1); */
#endif

#if (PCI_BASE_2_USED)
    pchip1[WSM0].r = PCI_MASK_2;
    pchip1[TBA0].r = PCI_TBASE_2;
    pchip1[WSBA0].r= PCI_BASE_2 | 1;

    /* WriteIOQ((((ul) PCHIP1_CSR<<24) | PCHIP1_WSM0), PCI_MASK_2); */
    /* WriteIOQ((((ul) PCHIP1_CSR<<24) | PCHIP1_TBA0), PCI_TBASE_2); */
    /* WriteIOQ((((ul) PCHIP1_CSR<<24) | PCHIP1_WSBA0), PCI_BASE_2 | 1); */
#endif


    /* Register information about the chipset? */

}





/*------------------------------------------------------------------------**
**                                                                        **
**  DP264 I/O procedures                                                  **
**                                                                        **
**      inport[b|w|l], outport[b|w|l] 8:16:32 IO xfers                    **
**                                                                        **
**      inmem[b|w|l], outmem[b|w|l] 8:16:32 PCI memory xfers              **
**                                                                        **
**------------------------------------------------------------------------*/



#define QW(d)	((ul)(d))

#define SWZ(d) ( \
		(((QW(d)>>(0*8))&0xff)<<(0*8)) | \
		(((QW(d)>>(1*8))&0xff)<<(2*8)) | \
		(((QW(d)>>(2*8))&0xff)<<(4*8)) | \
		(((QW(d)>>(3*8))&0xff)<<(6*8)) | \
		(((QW(d)>>(4*8))&0xff)<<(1*8)) | \
		(((QW(d)>>(5*8))&0xff)<<(3*8)) | \
		(((QW(d)>>(6*8))&0xff)<<(5*8)) | \
		(((QW(d)>>(7*8))&0xff)<<(7*8)))

#define UNSWZ(d) ( \
		(((QW(d)>>(0*8))&0xff)<<(0*8)) | \
		(((QW(d)>>(2*8))&0xff)<<(1*8)) | \
		(((QW(d)>>(4*8))&0xff)<<(2*8)) | \
		(((QW(d)>>(6*8))&0xff)<<(3*8)) | \
		(((QW(d)>>(1*8))&0xff)<<(4*8)) | \
		(((QW(d)>>(3*8))&0xff)<<(5*8)) | \
		(((QW(d)>>(5*8))&0xff)<<(6*8)) | \
		(((QW(d)>>(7*8))&0xff)<<(7*8)))



/*
 * The following macro define Type 0 and Type 1 configuration address format
 *
 *  4 4 4 4|3 3 3 3|3 3 3 3|3 3 2 2|2 2 2 2|2 2 2 2|1 1 1 1|1 1 1 1|1 1 
 *  3 2 1 0|9 8 7 6|5 4 3 2|1 0 9 8|7 6 5 4|3 2 1 0|9 8 7 6|5 4 3 2|1 0 9 8|7 6 5 4|3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |1|0|0|0|0|0|0|0|0|0|x|      0x1FE      |       bus     |  device |func.|  register |   |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *                                                                                       |
 *                                                                               Byte ---+
 */

/* NOTE: these macros assume that bus 0 and bus 1 directly map to
 * pchip 0 and pchip 1, which could well be wrong because we don't know 
 * whether any bridge devices are on either PCI bus.  Recipe for disaster!
 *
 * Currently all IO and memory accesses are erroneously put onto PCI bus 0,
 * bus 0 is on Pchip 0 and bus 1 is on Pchip 1 and there are no sub-buses!
 */

#define PCI0_CONFIG(bus,device,function,register) \
	((ul)(PCI0_CFG | (register & 0xff) | ((function & 0x7) << 8) \
	      | ((device & 0x1f) << 11) | ((bus & 0xff) << 16)))

#define PCI1_CONFIG(bus,device,function,register) \
	((ul)(PCI1_CFG | (register & 0xff) | ((function & 0x7) << 8) \
	      | ((device & 0x1f) << 11) | ((bus & 0xff) << 16)))

#define IO_ACCESS(p) ((ul)(PCI0_IO | p))
#define MEM_ACCESS(p) ((ul)(PCI0_MEM | p))


/* currently does nothing, disabling multiple hoses */
static unsigned bus2hose( const ui bus )
{
    /* paranoia: most basic case is when *nothing* is created yet? */
    if ( pci_hose[0] == NULL )		return 0;

    if ( (bus >= pci_hose[0]->number) && (bus <= pci_hose[0]->subordinate) )
					return 0;
    else 				return 1;	/* something else */
}

void pcicfgwb(ui bus, ui dev, ui func, ui reg, ui data)
{
    unsigned hose = bus2hose(bus);
    data &= 0xff;
#ifdef PCIDEBUG
    mobo_logf(LOG_DBG
	      "cfgwb hose %d bus %d dev %d func %d reg %d data %d\n", hose,
	      bus, dev, func, reg, data);
#endif
    if (hose == 0)
	WriteIOB(PCI0_CONFIG(bus, dev, func, reg), data);
    if (hose == 1) {
	if ( bus == pci_hose[1]->number )	bus = 0;
	WriteIOB(PCI1_CONFIG(bus, dev, func, reg), data);
    }
    mb();
}

void pcicfgww(ui bus, ui dev, ui func, ui reg, ui data)
{
    unsigned hose = bus2hose(bus);
    data &= 0xffff;
#ifdef PCIDEBUG
    mobo_logf(LOG_DBG
	      "cfgww hose %d bus %d dev %d func %d reg %d data %d\n", hose,
	      bus, dev, func, reg, data);
#endif
    if (hose == 0)
	WriteIOW(PCI0_CONFIG(bus, dev, func, reg), data);
    if (hose == 1) {
	if ( bus == pci_hose[1]->number )	bus = 0;
	WriteIOW(PCI1_CONFIG(bus, dev, func, reg), data);
    }
    mb();
}

void pcicfgwl(ui bus, ui dev, ui func, ui reg, ui data)
{
    unsigned hose = bus2hose(bus);
#ifdef PCIDEBUG
    mobo_logf(LOG_DBG
	      "cfgwl hose %d bus %d dev %d func %d reg %d data %d\n", hose,
	      bus, dev, func, reg, data);
#endif
    if (hose == 0)
	WriteIOL(PCI0_CONFIG(bus, dev, func, reg), data);
    if (hose == 1) {
	if ( bus == pci_hose[1]->number )	bus = 0;
	WriteIOL(PCI1_CONFIG(bus, dev, func, reg), data);
    }
    mb();
}

ui pcicfgrb(ui bus, ui dev, ui func, ui reg)
{
    unsigned hose = bus2hose(bus);
    ul result = ~0UL;
#ifdef PCIDEBUG
    mobo_logf(LOG_DBG "cfgrb hose %d bus %d dev %d func %d reg %d\n",
	      hose, bus, dev, func, reg);
#endif
    if (hose == 0)
	result = ReadIOB(PCI0_CONFIG(bus, dev, func, reg));
    if (hose == 1) {
	if ( bus == pci_hose[1]->number )	bus = 0;
	result = ReadIOB(PCI1_CONFIG(bus, dev, func, reg));
    }
    return (result & 0xff);
}

ui pcicfgrw(ui bus, ui dev, ui func, ui reg)
{
    unsigned hose = bus2hose(bus);
    ul result = ~0UL;
#ifdef PCIDEBUG
    mobo_logf(LOG_DBG "cfgrw hose %d bus %d dev %d func %d reg %d\n",
	      hose, bus, dev, func, reg);
#endif
    if (hose == 0)
	result = ReadIOW(PCI0_CONFIG(bus, dev, func, reg));
    if (hose == 1) {
	if ( bus == pci_hose[1]->number )	bus = 0;
	result = ReadIOW(PCI1_CONFIG(bus, dev, func, reg));
    }
    return (result & 0xffff);
}

ui pcicfgrl(ui bus, ui dev, ui func, ui reg)
{
    unsigned hose = bus2hose(bus);
    ul result = ~0UL;
#ifdef PCIDEBUG
    mobo_logf(LOG_DBG "cfgrl hose %d bus %d dev %d func %d reg %d\n",
	      hose, bus, dev, func, reg);
#endif
    if (hose == 0)
	result = ReadIOL(PCI0_CONFIG(bus, dev, func, reg));
    if (hose == 1) {
	if ( bus == pci_hose[1]->number )	bus = 0;
	result = ReadIOL(PCI1_CONFIG(bus, dev, func, reg));
    }
    return (result & 0xffffffffUL);
}
/* ----- TIG bus routines (Tsunami special) -------------*/
/* All TIG bus accesses are by quadword, but actually only a byte is used */

#define TIG_ACCESS(p) ((ul)(TIG_IO | (p<<6)))


void tigwb( ui offset, ub data )
{
    ul ldata = data & 0xFF;
    WriteIOQ( TIG_ACCESS( offset ), ldata );
    mb();
}

ub tigrb( ui offset )
{
    ul result;
    result = ReadIOQ( TIG_ACCESS( offset ) );
    return ( result & 0xFF );
}

/* ----- port routines -------------*/

void outportb(ui port,ui data)
{
  data&=0xff;
  WriteIOB(IO_ACCESS(port),data);
  mb();
}

void outportw(ui port,ui data)
{
  data&=0xffff;
  WriteIOW(IO_ACCESS(port),data);
  mb();
}

void outportl(ui port,ui data)
{
  WriteIOL(IO_ACCESS(port),data);
  mb();
}

void outportq(ui port, ul data)
{
  WriteIOQ(IO_ACCESS(port),data);
  mb();
}


ui inportb(ui port)
{
  ul result;
  result = ReadIOB(IO_ACCESS(port));
  return(result & 0xff);
}

ui inportw(ui port)
{
  ul result;
  result = ReadIOW(IO_ACCESS(port));
  return(result & 0xffff);
}

ui inportl(ui port)
{
  ul result;
  result = ReadIOL(IO_ACCESS(port));
  return(result & 0xffffffff);
}

ul inportq(ui port)
{
  return ReadIOQ(IO_ACCESS(port));
}

/* ----- memory routines -------------*/

void outmemb(ui port,ui data)
{
  data&=0xff;
  WriteIOB(MEM_ACCESS(port),data);
  mb();
}

void outmemw(ui port,ui data)
{
  data&=0xffff;
  WriteIOW(MEM_ACCESS(port),data);
  mb();
}

void outmeml(ui port,ui data)
{
  WriteIOL(MEM_ACCESS(port),data);
  mb();
}

ui inmemb(ui port)
{
  ub result;
  result = ReadIOB(MEM_ACCESS(port));
  return(result & 0xff);
}

ui inmemw(ui port)
{
  uw result;
  result = ReadIOW(MEM_ACCESS(port));
  return(result & 0xffff);
}

ui inmeml(ui port)
{
  ui result;
  result = ReadIOL(MEM_ACCESS(port));
  return(result & 0xffffffff);
}

/*------------------------------------------------*/

/* Note - there are two PCI buses (if a second Pchip is present) so we may need
 * to IACK devices on PCI bus 1 from time to time! */

ui inIack(void)
{
    return (*((ui*)PCI0_IACK) & 0xffU);
}



/*
 * Clear the NODEV status and
 * return the current value (TRUE/FALSE).
 */
ul IOPCIClearNODEV(void)
{
  return FALSE;
}


/*----------------------------------------------------------------------*/
/* SMP functionality */


/* On Tsunami systems, the CPU ID is read from the CPUID field, in the CChip
 * MISC<1:0> */

unsigned nb_whami( void )
{
    return cchip[MISC].r & 0x3;
}



/*----------------------------------------------------------------------*/
/* Error state analysis - produce meaningful interpretation of Tsunami state
 * on system machine check interrupts.
 */

/* Pchip_err_analyse_clear:
 * 
 * INPUTS:
 * Analysis	a string buffer to contain the interpretation output
 * P 		a pointer into Tsunami CSR space to the Pchip in question
 *
 * This function produces an error analysis and attempts to clear the error
 */

static const String ECCcmd[] = { "DMA read",
				 "DMA read-modify-write",
				 "Undefined operation(!)",
				 "Scatter-gather table read" };
#define UNDEFECCCMD	2
#define MAXECCCMD	3

void pchip_err_analyse_clear( String Analysis, unsigned pchipno )
{
    unsigned long error;
    unsigned long errbits, syn, cmd, inv, addr;
    tsunami_csr *P;

    if ( pchipno == 0 )		P = pchip0;
    else if ( pchipno == 1 )	P = pchip1;
    else {
	mobo_logf( LOG_CRIT "PCHIP: no such Pchip as %d!\n", pchipno );
	return;
    }

    /* The error register holds all the gory details */
    error = P[ PERROR ].r;
    errbits = error & perror_m_ebits;
    syn = (error & perror_m_syn) >> perror_s_syn;
    cmd = (error & perror_m_cmd) >> perror_s_cmd;
    if ( cmd > MAXECCCMD )	cmd = UNDEFECCCMD;		/* validation */
    inv = error & perror_m_inv;
    addr = (error & perror_m_addr) >> perror_s_addr;
    addr &= ~0x3UL;				/* remove extraneous details */
    

    switch( errbits )		/* only ONE errbit should be set */
    {
	case 0:
	    sprintf_dbm( Analysis, "No errors reported" );
	    break;

	case perror_m_serr:	/* b_serr_l asserted on bus */
	    sprintf_dbm( Analysis,
		    "SERR asserted by PCI device while Pchip %d was master\n"
		    "PCI address was 0x%08X", pchipno, addr );
	    break;

	case perror_m_perr:	/* b_perr_l asserted on bus */
	    sprintf_dbm( Analysis,
		    "PERR asserted by PCI device while Pchip %d was master\n"
		    "PCI address was 0x%08X", pchipno, addr );
	    break;

	case perror_m_dcrto:	/* delayed completion retry timeout */
	    sprintf_dbm( Analysis,
		    "Delayed-completion retry timeout at address 0x%08X\n"
		    "Pchip %d didn't get retry from other PCI device", 
			addr, pchipno );
	    break;

	case perror_m_sge:	/* scatter-gather invalide PTE */
	    sprintf_dbm( Analysis,
		    "Scatter-Gather DMA has invalid page table entry\n"
		    "PCI hose %d address 0x%08X", pchipno, addr );
	    break;

	case perror_m_ape:	/* Address parity error as potential target */
	    sprintf_dbm( Analysis,
			"A PCI device produced Address/Command parity error\n"
			"PCI hose %d address 0x%08X", pchipno, addr );
	    break;

	case perror_m_ta:	/* Target abort as PCI master */
	    sprintf_dbm( Analysis,
			"Pchip %d (as bus master) received target abort\n"
			"PCI bus address 0x%08X", pchipno, addr );
	    break;

	case perror_m_rdpe:	/* Read data parity error as PCI master */
	    sprintf_dbm( Analysis,
		"Pchip %d (as bus master) detected PIO read data parity error\n"
		"PCI bus address 0x%08X", pchipno, addr );
	    break;

	case perror_m_nds:	/* no b_devsel_l as PCI master */
	    sprintf_dbm( Analysis,
		"Pchip %d (as bus master) did not detect DEVSEL from target\n"
		"PCI bus address 0x%08X", pchipno, addr );
	    break;

	case perror_m_uecc:	/* uncorrectable ECC error from memory */
	case perror_m_cre:	/* correctable ECC error from memory */
	    sprintf_dbm( Analysis,
		"Pchip %d: %s ECC error from memory during %s\n"
		"Cache block address is 0x%016lX",
		pchipno,
		(errbits==perror_m_cre) ? "Correctable":"Uncorrectable",
		ECCcmd[cmd], addr );
	    break;

	default:
	    sprintf_dbm( Analysis, "Pchip %d: Bizarre unknown error 0x%03X",
			pchipno, errbits );
	    break;
    }

#if 0	/* This seems to happen an awful lot when there are no errors... */
    if ( inv )	mobo_logf( LOG_CRIT
			"PCHIP %d just gave data marked as invalid\n",
			pchipno );
#endif


    /* attempt to reset the error condition on the Pchip in question */
    /* this is done by writing to all error bits that are set high */

    P[ PERROR ].r = errbits;			/* errbits are W1C */
    mb();                                       /* make sure it goes */

}



/* CChip NXM analyse:
 * Under drastic circumstances NXMs may cause system machine check ints.
 * Here we take a look at the cchip MISC CSR to understand the source of 
 * such an NXM error.
 */

void cchip_nxm_analyse_clear( String interp )
{
    unsigned long misc;
    unsigned long nxs;

    misc = cchip[ MISC ].r;

    if ( (misc & misc_m_nxm) == 0 )
    {
	sprintf_dbm( interp, "CCHIP: didn't record any NXM error(??)" );
	return;
    }

    nxs = (misc & misc_m_nxs) >> misc_s_nxs;

    switch( nxs ) 
    {
	case misc_nxs_cpu0: 
	case misc_nxs_cpu1:
	    sprintf_dbm( interp,
			"CPU %d attempted to access non-existant memory",
			( nxs == misc_nxs_cpu0 ) ? 0 : 1 );
	    break;

	case misc_nxs_pchip0:
	case misc_nxs_pchip1:
	    sprintf_dbm( interp,
			"A device on PCI bus %d attempted to access "
			"non-existant memory",
			( nxs == misc_nxs_pchip0 ) ? 0 : 1 );
	    break;

	default:
	    sprintf_dbm( interp,
			"Unknown cause (%d) of non-existant memory error!",
			nxs );
	    break;
    }

    /* now attempt clearing of the NXM error state by writing 1 to MISC:NXM */
    cchip[ MISC ].r = misc_m_nxm;		/* write NXM bit to clear */
    mb();					/* make sure it goes */
}



